using System;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using Microsoft.Win32.SafeHandles;

namespace gov.va.med.vbecs.GUI.controls
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>3/18/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// Base class for VBECS/Admin application startup/single user mode locks.
	/// Encapsulates common functionality and shared lock tokens.
	/// </summary>

	#endregion

	public abstract class VbecsAppSyncLockBase : IDisposable
	{
		private const int GlobalSyncLockWaitTimeoutMs = 1000;
		
		/// <summary>
		/// Global access synchronization token for global mutex used to ensure 
		/// exclusive access to global mutexes when creating/checking locks.
		/// </summary>
		protected const string GlobalMutexAccessSyncToken = @"Global\gov.va.med.vbecs.GlobalAppLockSyncToken";

		/// <summary>
		/// Global VBECS application synchronization tokens (mutex name).
		/// Used to flag running VBECS application on the server. 
		/// Two are needed; one for test and another for prod.
		/// </summary>
		protected const string GlobalTestVbecsSyncToken = @"Global\gov.va.med.vbecs.GlobalTestVbecsSyncToken";

		/// <summary>
		/// GlobalProdVbecsSyncToken
		/// </summary>
		protected const string GlobalProdVbecsSyncToken = @"Global\gov.va.med.vbecs.GlobalProdVbecsSyncToken";
		
		/// <summary>
		/// Global VBECS Administrator application single user mode sychronization tokens (mutex name). 
		/// Used to flag VBECS Administrator application running single user mode.
		/// Two are needed; one for test and another for prod.
		/// </summary>
		protected const string GlobalAdminTestSingleUserModeSyncToken = @"Global\gov.va.med.vbecs.GlobalAdminTestSingleUserModeSyncToken";

		/// <summary>
		/// GlobalAdminProdSingleUserModeSyncToken
		/// </summary>
		protected const string GlobalAdminProdSingleUserModeSyncToken = @"Global\gov.va.med.vbecs.GlobalAdminProdSingleUserModeSyncToken";
 
		private SharedWin32Mutex _globalSyncMutex;
		private bool _isGlobalSyncMutexOwned;

		/// <summary>
		/// Base constructor for VBECS and Administrator application locks. 
		/// Obtains exclusive serverwide lock, creates/checks application specific locks 
		/// defined by the derived classes and then releases exclusive lock. 
		/// </summary>
		protected VbecsAppSyncLockBase()
		{
			_isGlobalSyncMutexOwned = false;

			try
			{
				ObtainGlobalLock();				

				ObtainAppSpecificLocks();
			}
			finally
			{
				lock( this )
					ReleaseGlobalLock();
			}
		}

		/// <summary>
		/// Creates/checks application-specific locks defined by derived classes. 
		/// </summary>
		protected abstract void ObtainAppSpecificLocks();

		/// <summary>
		/// Releases application-specific locks defined by derived classes. 
		/// </summary>
		protected abstract void ReleaseAppSpecificLocks();

		/// <summary>
		/// Nullifies application-specific locks defined by derived classes (sets member variables to nulls). 
		/// </summary>
		protected abstract void NullifyAppSpecificLocks();

		/// <summary>
		/// Obtains global synchronization lock to ensure exclusive access when creating/checking locks.
		/// </summary>
		private void ObtainGlobalLock()
		{
			_globalSyncMutex = new SharedWin32Mutex( GlobalMutexAccessSyncToken );
            SafeWaitHandle sh = _globalSyncMutex.SafeWaitHandle;
			if( !( _isGlobalSyncMutexOwned = _globalSyncMutex.WaitOne( GlobalSyncLockWaitTimeoutMs, false ) ) )
				throw( new BaseApplicationException( StrRes.SysErrMsg.Common.TimedOutWhileWaitingForGlobalSyncLock().ResString ) );
		}

		/// <summary>
		/// Releases global synchronization lock. 
		/// </summary>
		private void ReleaseGlobalLock()
		{			
			if( _globalSyncMutex == null )
				return;

			if( _globalSyncMutex == null )
				return;

			if( _isGlobalSyncMutexOwned )
				_globalSyncMutex.ReleaseMutex();

			CloseAndNullifyMutex( ref _globalSyncMutex );
			_isGlobalSyncMutexOwned = false;
		}

		/// <summary>
		/// Finalizes the instance.
		/// </summary>
		~VbecsAppSyncLockBase()
		{
			Dispose( false );
		}

		/// <summary>
		/// Disposes the object releasing all locks and performing the cleanup.
		/// </summary>
		public void Dispose()
		{
			GC.SuppressFinalize( this );
			Dispose( true );
		}

		private void Dispose( bool disposing )
		{				
			lock( this )
			{
				if( disposing )
				{
					ReleaseGlobalLock();			
					ReleaseAppSpecificLocks();
				}

				_globalSyncMutex = null;

				NullifyAppSpecificLocks();
			}
		}		

		/// <summary>
		/// Closes given mutex if it's not null and sets referenced variable to null.
		/// </summary>
		/// <param name="mutexToNullify">Reference to mutex variable.</param>
		protected void CloseAndNullifyMutex( ref SharedWin32Mutex mutexToNullify )
		{
			if( mutexToNullify == null )
				return;

			mutexToNullify.Close();
			mutexToNullify = null;
		}
	}
}
